/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.log4j.helpers;

import java.io.IOException;
import java.io.Writer;

import org.apache.log4j.spi.ErrorCode;
import org.apache.log4j.spi.ErrorHandler;

/**
 * Counts the number of characters written. Access to the count is synchronized
 * using the {@link Writer}'s lock. This is necessary because the count is
 * maintained as a <tt>long</tt>.
 * 
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.2
 * @see CountingQuietWriter
 */
public class SynchronizedCountingQuietWriter extends QuietWriter {

  private long charCount;

  public SynchronizedCountingQuietWriter(final Writer writer,
      final ErrorHandler eh) {
    super(writer, eh);
  }

  public void write(final String string) {
    try {
      synchronized (super.lock) {
        out.write(string);
        this.charCount += string.length();
      }
    } catch (IOException e) {
      errorHandler.error("Failed to write [" + string + "].", e,
          ErrorCode.WRITE_FAILURE);
    }
  }

  public long getCount() {
    synchronized (super.lock) {
      return charCount;
    }
  }

  public void setCount(final long count) {
    synchronized (super.lock) {
      this.charCount = count;
    }
  }
}
